#!/bin/sh
#
# Script de lancement des rgles de filtrage IPv6.
#
# chkconfig: 2345 08 92
#
# description: Firewall ipv6
#
# config: /etc/sysconfig/ip6tables

# Source 'em up
. /etc/init.d/functions

IP6TABLES_CONFIG=/etc/sysconfig/ip6tables

if [ ! -x /sbin/ip6tables ]; then
        exit 0
fi

# Interface pont IPv6
EXTIF=br0

# Prfixe local ( complter selon votre configuration
LOCAL=2xxx:xxxx:xxxx:xxxx::/64

# Toute adresse IPv6
ANY=::/0

# Adresses de lien local
LINK=fe80::/10

# Adresses multicast
MULTICAST=ff00::/8

# Chargement du module IPv6 du kernel si ce n'est pas dj fait
modprobe ipv6


function start
{
   action "Flushing all current rules and user defined chains:" true
   # Comportement par dfaut
   ip6tables -P INPUT DROP
   ip6tables -P OUTPUT ACCEPT
   ip6tables -P FORWARD DROP

   # Vider toutes les chaines
   ip6tables -t filter -F
   ip6tables -t mangle -F
   ip6tables -t filter -X       # Removing user defined chains

   echo -n "Applying ip6tables firewall rules: "

   # ********************************************************************
   # *                                                                  *
   # *          =   C h a i n e s   U t i l i s a t e u r   =           *
   # *                                                                  *
   # ********************************************************************

   # Chaine utilisateur (INDROP: Rception refuse, pas de rponse)
   ip6tables -t filter -N INDROP
   ip6tables -t filter -A INDROP -m limit --limit 1/s --limit-burst 10 -j LOG --log-level info --log-prefix "TENTATIVE InputDrop6 "
   ip6tables -t filter -A INDROP -j DROP

   # Chaine utilisateur (INREJECT: Rception refuse avec rponse)
   ip6tables -t filter -N INREJECT
   ip6tables -t filter -A INREJECT -m limit --limit 1/s --limit-burst 10 -j LOG --log-level info --log-prefix "TENTATIVE InputReject6 "
   ip6tables -t filter -A INREJECT -j REJECT

   # Chaine utilisateur (FWDROP: Forward refus, pas de rponse)
   ip6tables -t filter -N FWDROP
   ip6tables -t filter -A FWDROP -m limit --limit 1/s --limit-burst 10 -j LOG --log-level info --log-prefix "TENTATIVE ForwardDrop6 "
   ip6tables -t filter -A FWDROP -j DROP

   # Chaine utilisateur (FWREJECT: Forward refus avec rponse)
   ip6tables -t filter -N FWREJECT
   ip6tables -t filter -A FWREJECT -m limit --limit 1/s --limit-burst 10 -j LOG --log-level info --log-prefix "TENTATIVE ForwardReject6 "
   ip6tables -t filter -A FWREJECT -j REJECT

   # ********************************************************************
   # *                                                                  *
   # *               =   F i l t r e   I N P U T   =                    *
   # *                                                                  *
   # ********************************************************************

   # ICI LES REGLES DE FILTRAGE DU TRAFIC EXTERIEUR VERS LE FIREWALL

   # Trafic autoris en interne
   ip6tables -t filter -A INPUT -s ${LOCAL} -j ACCEPT
   ip6tables -t filter -A INPUT -s ${LINK} -j ACCEPT
   ip6tables -t filter -A INPUT -d ${MULTICAST} -j ACCEPT
   ip6tables -t filter -A INPUT -i lo -j ACCEPT

   # Seul le trafic UDP pour la rsolution DNS est autoris
   ip6tables -t filter -A INPUT -i $EXTIF -p udp -m udp --sport 53 -j ACCEPT
   ip6tables -t filter -A INPUT -i $EXTIF -p udp -j INDROP

   # Trafic TCP sur les ports < 1024 interdit
   ip6tables -t filter -A INPUT -i $EXTIF -p tcp -m tcp --dport 0:1023 -j INDROP

   # Accepter le trafic des connexions etablies
   ip6tables -t filter -A INPUT -i $EXTIF -m state --state ESTABLISHED,RELATED -j ACCEPT

   # Connexions TCP entrantes interdites
   ip6tables -t filter -A INPUT -i $EXTIF -p tcp -m tcp --tcp-flags SYN,RST,ACK SYN -j INDROP

   # Autorisations icmpv6 sauf ping entrant
   ip6tables -t filter -A INPUT -i $EXTIF -p icmpv6 --icmpv6-type echo-request -j INDROP
   ip6tables -t filter -A INPUT -i $EXTIF -p icmpv6 -j ACCEPT

   # Tout le reste est refus, on trace
   ip6tables -t filter -A INPUT -i $EXTIF -j INDROP

   # ********************************************************************
   # *                                                                  *
   # *               =   F i l t r e   O U T P U T   =                  *
   # *                                                                  *
   # ********************************************************************

   # ICI LES REGLES DE FILTRAGE DU TRAFIC EMIS PAR LE FIREWALL


   # ********************************************************************
   # *                                                                  *
   # *              =   F i l t r e   F O R W A R D   =                 *
   # *                                                                  *
   # ********************************************************************

   # ICI LES REGLES DE FILTRAGE DU TRAFIC TRAVERSANT LE FIREWALL

   # Retransmission des adresses locales
   ip6tables -t filter -A FORWARD -s ${LINK} -j ACCEPT

   # Retransmission des paquets ICMP sauf ping de l'internet vers l'intranet
   ip6tables -t filter -A FORWARD ! -s ${LOCAL} -p icmpv6 --icmpv6-type echo-request -j FWDROP
   ip6tables -t filter -A FORWARD -p icmpv6 -j ACCEPT

   # Accepter le trafic des connexions etablies
   ip6tables -t filter -A FORWARD -m state --state ESTABLISHED,RELATED -j ACCEPT

   # Accepter les flux sortants (a vous d'affiner)
   ip6tables -t filter -A FORWARD -s ${LOCAL} -j ACCEPT

   # Accepter les flux sortant locaux
   ip6tables -t filter -A FORWARD -s ::1 -j ACCEPT

   # Interdire tout le reste
   ip6tables -t filter -A FORWARD -j FWDROP

   # ********************************************************************
   # *                                                                  *
   # *       =   M o d i f i c a t i o n   d e   p a q u e t s   =      *
   # *                                                                  *
   # ********************************************************************

   # ICI LES REGLES DE MODIFICATION DU TRAFIC TRAVERSANT LE FIREWALL

   success "Applying ip6tables firewall rules"
   echo ""

   touch /var/lock/subsys/ip6tables
}

stop() {
        action "Flushing all chains:" ip6tables -F
        action "Removing user defined chains:" ip6tables -X
        echo $"Resetting built-in chains to the default ACCEPT policy:"
           ip6tables -P INPUT ACCEPT && \
           ip6tables -P FORWARD ACCEPT && \
           ip6tables -P OUTPUT ACCEPT && \
           ip6tables -t filter -F && \
           ip6tables -t mangle -F && \
           ip6tables -X && \
           success "Resetting built-in chains to the default ACCEPT policy" || \
           failure "Resetting built-in chains to the default ACCEPT policy"
        echo
        rm -f /var/lock/subsys/ip6tables
}

case "$1" in
  start)
        start
        ;;

  stop)
        stop
        ;;

  restart)
        start
        ;;

  condrestart)
        [ -e /var/lock/subsys/ip6tables ] && start
        ;;

  status)
        ip6tables --list
        ;;

  panic)
        echo $"Changing target policies to DROP: "
            ip6tables -P INPUT DROP && \
            ip6tables -P FORWARD DROP && \
            ip6tables -P OUTPUT DROP && \
            success "Changing target policies to DROP" || \
            failure "Changing target policies to DROP"
        echo
        action "Flushing all chains:" ip6tables -t filter -F && ip6tables -t mangle -F
        action "Removing user defined chains:" ip6tables -X
        ;;

  save)
        echo -n "Saving current rules to $IP6TABLES_CONFIG: "
        /sbin/ip6tables-save > $IP6TABLES_CONFIG  2>/dev/null && \
          success "Saving current rules to $IP6TABLES_CONFIG" || \
          failure "Saving current rules to $IP6TABLES_CONFIG"
        echo
        ;;

  *)
        echo "Usage: $0 {start|stop|restart|condrestart|status|panic|save}"
        exit 1
esac

exit 0
