/*
##########################################################################
##                                                                      ##
##   RENE GARCIA                          GNU General Public License V2 ##
##                                                                      ##
##########################################################################
##                                                                      ##
##      Projet  : NFQ_RAD Linux                                         ##
##      Module  : NFQ_RAD                                               ##
##      Piece   : nfq_rad.c                                             ##
##      Langage : C ANSI                                                ##
##      Auteur  : Ren GARCIA                                           ##
##                                                                      ##
##########################################################################
*/

    /*-
     *
     *  $Id: nfq_rad.c,v 1.1 2009/10/21 21:11:59 rgarcia Exp $
     *
     *  $Log: nfq_rad.c,v $
     *  Revision 1.1  2009/10/21 21:11:59  rgarcia
     *  Version initiale
     *
     *
    -*/

/***************************  Inclusions  *******************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <linux/netfilter.h>    /* for NF_ACCEPT */
#include <string.h>
#include <syslog.h>
#include <libnetfilter_queue/libnetfilter_queue.h>

/************************************************************************
*               unsigned short calcChecksumIPv6 (data, len)      PRIVE  *
*               ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~             *
* Objet : Calcul du checksum d'une trame IPv6                           *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  data : (const char *) Donnes de la trame                            *
*  len  : (int) Taille de la trame en octets                            *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (unsigned short) Checksum (calcul de CRC)                             *
*                                                                       *
************************************************************************/

static unsigned short
calcChecksumIPv6 (const char *data, int len)
{
   register int i;

   unsigned char tampon[65535]; /* Tempon temporaire */

   unsigned long checksum = 0;

   /* Copie des donnes dans le tampon temporaire */
   memcpy (tampon, data, len);

   /* -------  Creation du pseudo header IPv6 */
   /* IPv6 Source */
   memcpy (tampon + 0x00, data + 0x08, 16);
   /* IPv6 Destination */
   memcpy (tampon + 0x10, data + 0x18, 16);
   /* Taille des donnes utilisables */
   tampon[0x20] = 0x00;
   tampon[0x21] = 0x00;
   tampon[0x22] = 0x00;
   tampon[0x23] = len - 0x28;
   /* Type de trame (header suivant) */
   tampon[0x24] = 0x00;
   tampon[0x25] = 0x00;
   tampon[0x26] = 0x00;
   tampon[0x27] = data[6];

   /* RAZ du champ checksum */
   tampon[0x2A] = 0x00;
   tampon[0x2B] = 0x00;

   /* Calcul du checksum */
   for (i = 0; i < len; i += 2)
   {
      unsigned short w = ((tampon[i] << 8) & 0xFF00) + tampon[i + 1];

      checksum += (unsigned long) w;
   }

   /* Somme des retenues */
   while (checksum >> 16)
   {
      checksum = (checksum & 0xFFFF) + (checksum >> 16);
   }

   /* Complment  1 du checksum */
   checksum = ~checksum;

   return ((unsigned short) (checksum & 0xFFFF));
}

/************************************************************************
*                       u_int32_t  getPktId(nfq_data)           PRIVE   *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                   *
* Objet : Recuperation du numero d'ID de trame dans la queue            *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  nfq_data : (struct nfq_data *) Donnes de la trame                   *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (u_int32_t) Id de la trame dans la queue                              *
*                                                                       *
************************************************************************/

/* returns packet id */
static u_int32_t
getPktId (struct nfq_data *tb)
{
   u_int32_t id = 0;

   struct nfqnl_msg_packet_hdr *ph;

   ph = nfq_get_msg_packet_hdr (tb);
   if (ph)
   {
      id = ntohl (ph->packet_id);
   }

   return id;
}

/************************************************************************
*                       int callbackRad(qh,nfmsg,nfa,trame)     PRIVE   *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~             *
* Objet : Callback appele lors la reception d'une trame                *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  qh : (struct nfq_q_handle *) Descripteur de file                     *
*  nfmsg : (struct nfgenmsg *) Descripteur de message dans la file      *
*  nfa : (struct nfq_data *) Contexte                                   *
*  trame : (void *) Copie de la trame                                   *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout c'est bien pass                                      *
*                                                                       *
************************************************************************/

static int
callbackRad (struct nfq_q_handle *qh, struct nfgenmsg *nfmsg,
             struct nfq_data *nfa, void *trame)
{
   unsigned char *data;
   unsigned long len;

   u_int32_t id = getPktId (nfa);

   len = nfq_get_payload (nfa, (char **) &data);

   if (len > 0)
   {
      unsigned short checksum;

      /* Modification des flags */
      data[0x2D] |= 0xC0;

      /* Recalcul du checksum */
      checksum = calcChecksumIPv6 (data, len);
      data[0x2A] = (checksum >> 8) & 0xFF;
      data[0x2B] = checksum & 0xFF;
   }
   else
   {
      data = NULL;
   }

   return nfq_set_verdict (qh, id, NF_ACCEPT, len, data);
}

/************************************************************************
***                                                                   ***
**            =   F O N C T I O N   P R I N C I P A L E   =            **
***                                                                   ***
************************************************************************/

int
main (int argc, char **argv)
{
   struct nfq_handle *h;
   struct nfq_q_handle *qh;
   struct nfnl_handle *nh;

   int fd;
   int rv;
   int queue;

   char buf[4096] __attribute__ ((aligned));

   if (argc != 2)
   {
      fprintf (stderr, "USAGE: %s queue_num\n", argv[0]);
      exit (1);
   }
   else
   {
      queue = atoi (argv[1]);
      if (!queue)
      {
         fprintf (stderr, "queue_num must be a number greater than 0\n");
         exit (1);
      }
   }

   h = nfq_open ();
   if (!h)
   {
      perror ("nfq_open");
      fprintf (stderr, "error during nfq_open()\n");
      exit (1);
   }

   if (nfq_unbind_pf (h, PF_INET6) < 0)
   {
      perror ("nfq_unbind_pf");
      fprintf (stderr, "error during nfq_unbind_pf()\n");
      exit (1);
   }

   if (nfq_bind_pf (h, PF_INET6) < 0)
   {
      perror ("nfq_bind_pf");
      fprintf (stderr, "error during nfq_bind_pf()\n");
      exit (1);
   }

   qh = nfq_create_queue (h, queue, &callbackRad, NULL);
   if (!qh)
   {
      perror ("nfq_create_queue");
      fprintf (stderr, "error during nfq_create_queue()\n");
      exit (1);
   }

   if (nfq_set_mode (qh, NFQNL_COPY_PACKET, 0xffff) < 0)
   {
      perror ("nfq_set_mode");
      fprintf (stderr, "can't set packet_copy mode\n");
      exit (1);
   }

   if (daemon (0, 0) == -1)
   {
      perror ("daemon");
      fprintf (stderr, "can't detach process\n");
      exit (1);
   }

   syslog (LOG_NOTICE, "IPv6 router announce packet modifier starting");

   fd = nfq_fd (h);

   while ((rv = recv (fd, buf, sizeof (buf), 0)) && rv >= 0)
   {
      syslog (LOG_INFO, "IPv6 router announce packet modified");
      nfq_handle_packet (h, buf, rv);
   }

   syslog (LOG_NOTICE, "IPv6 router announce packet modifier exiting");
   nfq_destroy_queue (qh);

#ifdef INSANE
   /* normally, applications SHOULD NOT issue this command, since
    * it detaches other programs/sockets from PF_INET6, too ! */
   nfq_unbind_pf (h, PF_INET6);
#endif

   nfq_close (h);
   exit (0);
}

/* ######################  FIN DE FICHIER  ########################### */
