/*
##########################################################################
##                                                                      ##
##   RENE GARCIA                          GNU General Public License V2 ##
##                                                                      ##
##########################################################################
##                                                                      ##
##      Projet  : TEMPerHum Linux                                       ##
##      Module  : DS75                                                  ##
##      Piece   : DS75.c                                                ##
##      Langage : C ANSI                                                ##
##      Auteur  : Ren GARCIA                                           ##
##                                                                      ##
##########################################################################
*/

    /*-
     *
     *  $Id: DS75.c,v 1.1 2009/10/09 08:03:24 rgarcia Exp $
     *
     *  $Log: DS75.c,v $
     *  Revision 1.1  2009/10/09 08:03:24  rgarcia
     *  Version initiale
     *
     *
    -*/

/***************************  Inclusions  *******************************/

#include "I2C.h"
#include "DS75.h"

/************************************************************************
*                                                                       *
*               C O N S T A N T E S   D U   M O D U L E                 *
*                                                                       *
************************************************************************/

/* =======  Identifiant du chip */
#define DS75_ID     0x90

/* =======  Pointeur de donne */
#define DS75_TEMPERATURE      0
#define DS75_CONFIGURATION    1
#define DS75_THYST            2
#define DS75_TOS              3

/* =======  Type d'accs */
#define DS75_READ   1
#define DS75_WRITE  0

/************************************************************************
*                                                                       *
*            P R O T O T Y P E S   D E S   F O N C T I O N S            *
*                                                                       *
************************************************************************/

static int DS75_Mesure (unsigned char, unsigned short *);

/************************************************************************
*                  char DS75_ReadStatusReg(addr,pVal)            PRIVE  *
*                  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                   *
* Objet : Lecture du registre d'tat de la sonde DS75                   *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  addr : (unsigned char) Adresse de la sonde sur le bus I2C (0-7)      *
*  pVal : (unsigned char *) Pointeur sur l'octet qui recevra la valeur  *
*         du registre d'tat                                            *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout est OK, autre valeur sinon                            *
*                                                                       *
************************************************************************/

int
DS75_ReadStatusReg (unsigned char addr, unsigned char *pValue)
{
   /* =======  Variables locales */
   int status = 0;

   /* =======  Debut du code */

   /* Selection du registre de configuration */
   status +=
      I2C_Write16 (((DS75_ID | (addr << 1) | DS75_WRITE) << 8) |
                   DS75_CONFIGURATION);

   /* Lecture du registre de configuration */
   *pValue = I2C_Read8 (DS75_ID | (addr << 1) | DS75_READ);

   /* Code retour */
   return status;
}

/************************************************************************
*                char DS75_WriteStatusReg(addr,Value)            PRIVE  *
*                ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                   *
* Objet : Ecriture du registre d'tat de la sonde DS75                  *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  addr : (unsigned char) Adresse de la sonde sur le bus I2C (0-7)      *
*  Value: (unsigned char) Valeur  crire dans le registre d'tat       *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout est OK, autre valeur sinon                            *
*                                                                       *
************************************************************************/

int
DS75_WriteStatusReg (unsigned char addr, unsigned char Value)
{
   /* =======  Variables locales */
   int status = 0;

   /* =======  Debut du code */

   /* Selection du registre de configuration */
   status +=
      I2C_Write24 (((DS75_ID | (addr << 1) | DS75_WRITE) << 16) |
                   (DS75_CONFIGURATION << 8) | Value);

   /* Code retour */
   return status;
}

/************************************************************************
*                       int DS75_Mesure(addr,pVal)               PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~                      *
* Objet : Lecture d'une mesure la sonde DS75 (valeurs brutes)           *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  addr : (unsigned char) Adresse de la sonde sur le bus I2C (0-7)      *
*  pVal : (unsigned short *) Pointeur sur le mot qui recevra la valeur  *
*         de la mesure                                                  *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout est OK, autre valeur sinon                            *
*                                                                       *
************************************************************************/

static int
DS75_Mesure (unsigned char addr, unsigned short *pVal)
{
   /* =======  Variables locales */
   int status = 0;

   /* =======  Debut du code */

   /* Selection du registre de temperature */
   status +=
      I2C_Write16 (((DS75_ID | (addr << 1) | DS75_WRITE) << 8) |
                   DS75_TEMPERATURE);

   /* Lecture du registre de temperature */
   *pVal = I2C_Read16 (DS75_ID | (addr << 1) | DS75_READ);

   /* Code retour */
   return status;
}

/************************************************************************
*                   char DS75_GetTempHum(addr,pTemp)            PUBLIC  *
*                   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                    *
* Objet : Lecture d'une mesure de temperature de la sonde DS75          *
*         (valeur physique en degres centigrade)                        *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  addr : (unsigned char) Adresse de la sonde sur le bus I2C (0-7)      *
*  pTemp : (double *) Pointeur qui recevra la mesure de temperature     *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout s'est bien pass, autre valeur sinon                  *
*                                                                       *
************************************************************************/

int
DS75_GetTemp (unsigned char addr, double *pTemp)
{
   /* =======  Variables locales */
   short temp_val;

   unsigned int status = 0,
      i;

   /* =======  Debut du code */

   /* Lecture du capteur d'humidite et de temperature */
   status += DS75_Mesure (addr, (unsigned short *) &temp_val);

   if (!status)
   {
      /* Conversion de la valeur du capteurs en valeur physique */
      *pTemp = temp_val * 0.00390625;
   }

   /* Code de retour */
   return status;
}

/* ######################  FIN DE FICHIER  ########################### */
