/*
##########################################################################
##                                                                      ##
##   RENE GARCIA                          GNU General Public License V2 ##
##                                                                      ##
##########################################################################
##                                                                      ##
##      Projet  : TEMPerHum Linux                                       ##
##      Module  : rs232                                                 ##
##      Piece   : rs232.c                                               ##
##      Langage : C ANSI                                                ##
##      Auteur  : Ren GARCIA                                           ##
##                                                                      ##
##########################################################################
*/

    /*-
     *
     *  $Id: rs232.c,v 1.1 2009/10/09 08:03:24 rgarcia Exp $
     *
     *  $Log: rs232.c,v $
     *  Revision 1.1  2009/10/09 08:03:24  rgarcia
     *  Version initiale
     *
     *
    -*/

/***************************  Inclusions  *******************************/

#include <sys/ioctl.h>
#include <fcntl.h>
#include <termios.h>
#include <string.h>
#include <stdlib.h>
#include "rs232.h"

/************************  Definitions locales  *************************/

#define BAUDRATE B9600
#define MODEMDEVICE "/dev/ttyUSB0"

/************************************************************************
*                                                                       *
*        V A R I A B L E S   G L O B A L E S   A U   M O D U L E        *
*                                                                       *
************************************************************************/

/* Descripteur de port serie */
static int fd;

/* Ancienne configuration du port */
static struct termios oldtio;

/************************************************************************
*                                                                       *
*            P R O T O T Y P E S   D E S   F O N C T I O N S            *
*                                                                       *
************************************************************************/

void rs232_DTR (register int set);
void rs232_RTS (register int set);
int rs232_CTS (void);

/************************************************************************
*                       void rs232_DTR(int set)                 PUBLIC  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~                         *
* Objet : Positionne la sortie DTR de la liaison serie a la valeur      *
*         souhaite  (CLK du bus I2C)                                   *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* set: (int) 1 pour activer la sortie, 0 pour la dsactiver             *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune) mais quitte le programme en cas d'erreur d'accs au port     *
*                                                                       *
************************************************************************/

void
rs232_DTR (register int set)
{
   /* =======  Variables locales */
   int status = 0;              /* Etat du port serie */

   /* =======  Debut du code */

   /* Recuperation de l'etat actuel du port */
   if (ioctl (fd, TIOCMGET, &status) < 0)
   {
      perror ("ioctl");
      exit (1);
   }

   /* Pose ou dpose le signal DTR */
   if (set)
   {
      status |= TIOCM_DTR;
   }
   else
   {
      status &= ~TIOCM_DTR;
   }

   /* Ecriture du nouvel etat du port */
   if (ioctl (fd, TIOCMSET, &status) < 0)
   {
      perror ("ioctl");
      exit (1);
   }
}

/************************************************************************
*                         void rs232_RTS(int set)               PUBLIC  *
*                         ~~~~~~~~~~~~~~~~~~~~~~~                       *
* Objet : Positionne la sortie RTS de la liaison serie a la valeur      *
*         souhaite  (SDA Master du bus I2C)                            *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* set: (int) 1 pour activer la sortie, 0 pour la dsactiver             *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune) mais quitte le programme en cas d'erreur d'accs au port     *
*                                                                       *
************************************************************************/

void
rs232_RTS (register int set)
{
   /* =======  Variables locales */
   int status = 0;              /* Etat du port serie */

   /* =======  Debut du code */

   /* Recuperation de l'etat actuel du port */
   if (ioctl (fd, TIOCMGET, &status) < 0)
   {
      perror ("ioctl");
      exit (1);
   }

   /* Pose ou dpose le signal RTS */
   if (set)
   {
      status |= TIOCM_RTS;
   }
   else
   {
      status &= ~TIOCM_RTS;
   }

   /* Ecriture du nouvel etat du port */
   if (ioctl (fd, TIOCMSET, &status) < 0)
   {
      perror ("ioctl");
      exit (1);
   }
}

/************************************************************************
*                            int rs232_CTS()                    PUBLIC  *
*                            ~~~~~~~~~~~~~~~                            *
* Objet : Lecture de l'entre CTS du port srie  (SDA Reel du bus I2C)  *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si le signal CTS n'est pas positionn, autre valeur sinon     *
*                                                                       *
************************************************************************/

int
rs232_CTS (void)
{
   /* =======  Variables locales */
   int status = 0;              /* Etat du port serie */

   /* =======  Debut du code */

   /* Recuperation de l'etat actuel du port */
   ioctl (fd, TIOCMGET, &status);

   /* Renvoie l'tat de la ligne CTS */
   return (status & TIOCM_CTS) ? 1 : 0;
}

/************************************************************************
*                            void rs232_Open()                  PUBLIC  *
*                            ~~~~~~~~~~~~~~~~~                          *
* Objet : Initialisation du controleur rs232 pour une utilisation I2C   *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
rs232_Open (void)
{
   /* =======  Variables locales */
   struct termios newtio;       /* Nouvelle configuration du port */

   /* =======  Debut du code */

   /* -------  Ouverture du port srie */
   fd = open (MODEMDEVICE, O_RDWR | O_NOCTTY);
   if (fd < 0)
   {
      perror (MODEMDEVICE);
      exit (-1);
   }

   /* -------  Lecture de la configuration du port pour sauvegarde */
   tcgetattr (fd, &oldtio);

   /* Copie de la configuration */
   memcpy (&newtio, &oldtio, sizeof (newtio));

   /* -------  Reconfiguration du port */
   newtio.c_cflag |= (CLOCAL | CREAD);

   newtio.c_cflag &= ~PARENB;
   newtio.c_cflag &= ~CSTOPB;
   newtio.c_cflag &= ~CSIZE;
   newtio.c_cflag |= CS8;

   cfsetispeed (&newtio, BAUDRATE);
   cfsetospeed (&newtio, BAUDRATE);

   newtio.c_cflag |= CRTSCTS;
   newtio.c_iflag |= IGNPAR;
   newtio.c_oflag &= ~OPOST;

   /* Parametrage du mode d'entree/sortie du port */
   newtio.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG);
   newtio.c_cc[VTIME] = 1;
   newtio.c_cc[VMIN] = 1;

   /* Vider le tampon d'entree */
   tcflush (fd, TCIFLUSH);

   /* Appliquer la nouvelle parametrie */
   tcsetattr (fd, TCSANOW, &newtio);
}

/************************************************************************
*                            void rs232_Close()                 PUBLIC  *
*                            ~~~~~~~~~~~~~~~~~~                         *
* Objet : Fermeture du controleur de bus rs232                          *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
rs232_Close (void)
{
   /* Restaurer l'ancienne parametrie du port srie */
   tcsetattr (fd, TCSANOW, &oldtio);
   close (fd);
}

/* ######################  FIN DE FICHIER  ########################### */
