/*
##########################################################################
##                                                                      ##
##   RENE GARCIA                          GNU General Public License V2 ##
##                                                                      ##
##########################################################################
##                                                                      ##
##      Projet  : TEMPerHum Linux                                       ##
##      Module  : I2C                                                   ##
##      Piece   : I2C.c                                                 ##
##      Langage : C ANSI                                                ##
##      Auteur  : Ren GARCIA                                           ##
##                                                                      ##
##########################################################################
*/

    /*-
     *
     *  $Id: I2C.c,v 1.1 2009/10/09 08:03:24 rgarcia Exp $
     *
     *  $Log: I2C.c,v $
     *  Revision 1.1  2009/10/09 08:03:24  rgarcia
     *  Version initiale
     *
     *
    -*/

/***************************  Inclusions  *******************************/

#include "rs232.h"
#include "I2C.h"

/************************************************************************
*                                                                       *
*        V A R I A B L E S   G L O B A L E S   E X P O R T E E S        *
*                                                                       *
************************************************************************/

/* Etat du controleur I2C */
static char I2C_CHic;

/************************************************************************
*                                                                       *
*            P R O T O T Y P E S   D E S   F O N C T I O N S            *
*                                                                       *
************************************************************************/

static void Delay (register int msec);
static void I2C_StartCond (void);
static void I2C_StopCond (void);

/************************************************************************
*                            void Delay(msec)                    PRIVE  *
*                            ~~~~~~~~~~~~~~~~                           *
* Objet : Temporisation                                                 *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* msec: (int)  Delai en centimes de milliseconde                       *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/


static void
Delay(register int msec)
{
    usleep(50 * msec);
}


/*#define Delay(x)*/

/************************************************************************
*                              void I2C_Sclk(set)               PUBLIC  *
*                              ~~~~~~~~~~~~~~~~~~                       *
* Objet : Positionne la ligne d'horloge SCL au niveau souhait          *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* set: (int) 1 ou 0                                                     *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
I2C_Sclk (register int a)
{
   /* Mode T du bus I2C : horloge normale */
   if (I2C_CHic == 'T')
   {
      rs232_DTR (a == 0);
   }
   /* Mode R du bus I2C : horloge inverse */
   else if (I2C_CHic == 'R')
   {
      rs232_DTR (a);
   }
}

/************************************************************************
*                           void I2C_SDout(set)                 PUBLIC  *
*                           ~~~~~~~~~~~~~~~~~~~                         *
* Objet : Positionne la ligne de donnes SDA au niveau souhait         *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* bit: (int) 1 ou 0                                                     *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
I2C_SDout (register int a)
{
   /* Mode T du bus I2C : bus de donnes normal */
   if (I2C_CHic == 'T')
   {
      rs232_RTS (a == 0);
   }
   /* Mode R du bus I2C : bus de donnes invers */
   else if (I2C_CHic == 'R')
   {
      rs232_RTS (a);
   }
}

/************************************************************************
*                               int I2C_SDin()                  PUBLIC  *
*                               ~~~~~~~~~~~~~~                          *
* Objet : Lecture du niveau de la ligne de donnes SDA du bus I2C       *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 ou 1 selon la valeur du bit lu                                *
*                                                                       *
************************************************************************/

int
I2C_SDin (void)
{
   /* =======  Variables locales */
   register int a = 0;

   /* =======  Debut du code */

   /* Lecture de la ligne d'entre */
   a = rs232_CTS ();

   /* Mode T du bus I2C : bus de donnes normal */
   if (I2C_CHic == 'T')
   {
      a = (a ? 0 : 1);
   }
   /* Mode R du bus I2C : bus de donnes invers */
   else if (I2C_CHic == 'R')
   {
      a = (a ? 1 : 0);
   }

   /* Valeur de retour */
   return a;
}

/************************************************************************
*                           void I2C_StartCond()                 PRIVE  *
*                           ~~~~~~~~~~~~~~~~~~~~                        *
* Objet : Envoie une condition de dpart sur le bus I2C                 *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

static void
I2C_StartCond (void)
{
    /*-
     *
     *  BUS I2C : La condition de dpart.
     *  SDA passe  '0' alors que SCL reste  '1'
     *
    -*/

   /* Data=1 Clk=? */
   I2C_SDout (1);
   Delay (4);

   /* Data=1 Clk=1 */
   I2C_Sclk (1);
   Delay (40);

   /* Data=0 Clk=1 */
   I2C_SDout (0);
   Delay (30);

   /* Data=0 Clk=0 */
   I2C_Sclk (0);
}

/************************************************************************
*                            void I2C_StopCond()                 PRIVE  *
*                            ~~~~~~~~~~~~~~~~~~~                        *
* Objet : Envoie une condition d'arret sur le bus I2C                   *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

static void
I2C_StopCond (void)
{
    /*-
     *
     *  BUS I2C : La condition d'arrt.
     *  SDA passe  '1' alors que SCL reste  '1'
     *
    -*/

   /* Data=0 Clk=? */
   I2C_SDout (0);
   Delay (50);

   /* Data=0 Clk=1 */
   I2C_Sclk (1);
   Delay (50);

   /* Data=1 Clk=1 */
   I2C_SDout (1);
   Delay (50);
}

/************************************************************************
*                             void I2C_HiLowSCLK()              PUBLIC  *
*                             ~~~~~~~~~~~~~~~~~~~~                      *
* Objet : Realise un cycle d'horloge                                    *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
I2C_HiLowSCLK (void)
{
   /* Tempo */
   Delay (10);

   /* Etat haut */
   I2C_Sclk (1);
   Delay (20);

   /* Etat bas */
   I2C_Sclk (0);
   Delay (20);
}

/************************************************************************
*                             void I2C_GetACK()                  PRIVE  *
*                             ~~~~~~~~~~~~~~~~~                         *
* Objet : Lecture de l'acquitement d'criture                           *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si acquittement OK, autre valeur sinon                        *
*                                                                       *
************************************************************************/

int
I2C_GetACK (void)
{
   register int ack;

    /*-
     *
     *  Bus I2C : Le bit ACK.
     *  Le maitre pose le bus SDA  1, l'esclave concern doit
     *  rpondre en plaant le bus SDA  0. Le maitre doit donc lire 0
     *  sur le bus. Si la ligne SDA est  1 le matre comprend qu'il y
     *  a une erreur de slection et doit poser la condition d'arret.
     *
    -*/

   /* Met SDA  1 */
   I2C_SDout (1);

   /* Tempo */
   Delay (10);

   /* Horloge a l'tat haut */
   I2C_Sclk (1);
   Delay (200);

   /* Lecture de la ligne de donnes */
   ack = I2C_SDin ();

   /* Horloge  l'tat bas */
   I2C_Sclk (0);
   Delay (20);

   /* Si la ligne de donnes tait  1, aquittement en erreur */

   if (ack)
   {
      /* On inverse la ligne, l'tat lectrique n'est peut-tre pas celui que l'on croit */
      if (I2C_CHic == 'T')
      {
         I2C_CHic = 'R';
      }
      else
      {
         I2C_CHic = 'T';
      }

      /* sortie avec erreur */
      return 1;
   }

   /* sortie OK */
   return 0;
}

/************************************************************************
*                          void I2C_SendACK(ack)                 PRIVE  *
*                          ~~~~~~~~~~~~~~~~~~~~~                        *
* Objet : Envoi de l'acquitement de lecture                             *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* ack : (unsigned char) 1 pour envoyer un acquittement, 0 sinon         *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
I2C_SendACK (unsigned char ack)
{
    /*-
     *
     *  Bus I2C : Le bit ACK.
     *  Aprs une lecture, le maitre pose le bus SDA  0 et genre un
     *  cycle d'horloge. L'esclave peut lire alors l'acquittement.
     *
    -*/

   /* Pose ou pas l'acquittement */
   I2C_SDout (!ack);

   /* 1 cycle d'horloge */
   I2C_HiLowSCLK ();

   /* Liberation de la ligne data */
   I2C_SDout (1);
}

/************************************************************************
*                         void I2C_RawWrite8(data)              PUBLIC  *
*                         ~~~~~~~~~~~~~~~~~~~~~~~~~                     *
* Objet : Ecriture d'un octet sur le bus de donnes I2C                 *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* datra : (unsigned char) octet  envoyer                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
I2C_RawWrite8 (register unsigned char data)
{
   /* =======  Variable locale */
   register unsigned char i;

   /* =======  Debut du code */

   /* Pour chacun des 8 bits */
   for (i = 0x80; i; i >>= 1)
   {
      /* Envoi du bit de poids fort */
      I2C_SDout (data & i);

      /* Temporisation */
      Delay (10);

      /* 1 cycle d'horloge */
      I2C_HiLowSCLK ();
   }
}

/************************************************************************
*                           data I2C_RawRead8()                 PUBLIC  *
*                           ~~~~~~~~~~~~~~~~~~~                         *
* Objet : Lecture d'un octet sur le bus de donnes I2C                  *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (unsigned char) valeur lue                                            *
*                                                                       *
************************************************************************/

unsigned char
I2C_RawRead8 (void)
{
   /* =======  Variable locale */
   register unsigned char data = 0;
   register unsigned char i;

   /* =======  Debut du code */

   /* Met SDA  1 */
   I2C_SDout (1);

   /* Tempo */
   Delay (10);

   /* Pour chacun des 8 bits */
   for (i = 0x80; i; i >>= 1)
   {
      /* Horloge a l'tat haut */
      I2C_Sclk (1);
      Delay (20);

      /* Lecture de la ligne de donnes */
      if (I2C_SDin ())
         data |= i;

      /* Horloge  l'tat bas */
      I2C_Sclk (0);
      Delay (20);
   }

   return data;
}

/************************************************************************
*                            int I2C_Write8(data)               PUBLIC  *
*                            ~~~~~~~~~~~~~~~~~~~~                       *
* Objet : Envoi d'une donne 8 bit sur le bus de donnes I2C            *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* data : (unsigned char) octet  envoyer                                *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si l'octet a t envoy correctement, autre valeur sinon      *
*                                                                       *
************************************************************************/

int
I2C_Write8 (unsigned char data)
{
   /* =======  Variables locales */
   int result;

   /* =======  Debut du code */

   /* Prise du bus */
   I2C_StartCond ();

   /* Envoi des 8 bits de l'octet */
   I2C_RawWrite8 (data);

   /* Demande d'acquittement */
   result = I2C_GetACK ();

   /* Liberation du bus */
   I2C_StopCond ();

   return result;
}

/************************************************************************
*                            int I2C_Write16(data)              PUBLIC  *
*                            ~~~~~~~~~~~~~~~~~~~~~                      *
* Objet : Envoi d'une donne 16 bit sur le bus de donnes I2C           *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* data : (unsigned short) donne  envoyer                              *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si la donne a t envoy correctement, autre valeur sinon    *
*                                                                       *
************************************************************************/

int
I2C_Write16 (unsigned short data)
{
   /* =======  Variables locales */
   int result;
   unsigned char msb = (data >> 8) & 0xFF;
   unsigned char lsb = data & 0xFF;

   /* =======  Debut du code */

   /* Prise du bus */
   I2C_StartCond ();

   /* Envoi des 8 bits de l'octet de poids fort */
   I2C_RawWrite8 (msb);

   /* Demande d'acquittement */
   result = I2C_GetACK ();

   /* On continue a lire les 8 bits suivants si l'acquittement est OK */
   if (!result)
   {
      /* Envoi des 8 bits de l'octet de poids faible */
      I2C_RawWrite8 (lsb);

      /* Demande d'acquittement */
      result = I2C_GetACK ();
   }

   /* Liberation du bus */
   I2C_StopCond ();

   return result;
}

/************************************************************************
*                            int I2C_Write24(data)              PUBLIC  *
*                            ~~~~~~~~~~~~~~~~~~~~~                      *
* Objet : Envoi d'une donne 24 bit sur le bus de donnes I2C           *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* data : (unsigned long) donne  envoyer                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si la donne a t envoy correctement, autre valeur sinon    *
*                                                                       *
************************************************************************/

int
I2C_Write24 (unsigned long data)
{
   /* =======  Variables locales */
   int result;
   unsigned char hsb = (data >> 16) & 0xFF;
   unsigned char msb = (data >> 8) & 0xFF;
   unsigned char lsb = data & 0xFF;

   /* =======  Debut du code */

   /* Prise du bus */
   I2C_StartCond ();

   /* Envoi des 8 bits de l'octet de poids fort */
   I2C_RawWrite8 (hsb);

   /* Demande d'acquittement */
   result = I2C_GetACK ();

   /* Acquittement OK, on continue pour 8 bits */
   if (!result)
   {
      /* Envoi des 8 bits de l'octet de poids moyen */
      I2C_RawWrite8 (msb);

      /* Demande d'acquittement */
      result = I2C_GetACK ();

      /* Acquittement OK, on continue pour 8 bits */
      if (!result)
      {
         /* Envoi des 8 bits de l'octet de poids faible */
         I2C_RawWrite8 (lsb);

         /* Demande d'acquittement */
         result = I2C_GetACK ();
      }
   }

   /* Liberation du bus */
   I2C_StopCond ();

   return result;
}

/************************************************************************
*                     unsigned char I2C_Read8(addr)             PUBLIC  *
*                     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                     *
* Objet : Lecture d'un octet  une adresse                              *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  addr : (unsigned char) Adresse 8 bit  lire                          *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (unsigned char) octet lu                                              *
*                                                                       *
************************************************************************/

unsigned char
I2C_Read8 (unsigned char addr)
{
   /* =======  Variables lcoales */
   register unsigned char data = 0;     /* Valeur lue */
   register int result;         /* resultat */

   /* Prise du bus */
   I2C_StartCond ();

   /* Envoi de l'adresse */
   I2C_RawWrite8 (addr);

   /* Demande d'acquittement */
   result = I2C_GetACK ();

   /* On continue sur une lecture si l'acquittement est OK */
   if (!result)
   {
      /* Lecture des 8 bits */
      data = I2C_RawRead8 ();

      /* Envoie un acquittement de lecture */
      I2C_SendACK (0);
   }

   /* Fin de communication I2C */
   I2C_StopCond ();

   /* Valeur de retour */
   return data;
}

/************************************************************************
*                    unsigned short I2C_Read16(addr)            PUBLIC  *
*                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                    *
* Objet : Lecture d'un mot de 16 bits  une adresse                     *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  addr : (unsigned char) Adresse 8 bit  lire                          *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (unsigned short) mot lu                                               *
*                                                                       *
************************************************************************/

unsigned short
I2C_Read16 (unsigned char addr)
{
   /* =======  Variables lcoales */
   register unsigned char msb = 0;      /* Valeur de poid fort lue */
   register unsigned char lsb = 0;      /* Valeur de poid faible lue */
   register int result;         /* Resultat */

   /* Prise du bus */
   I2C_StartCond ();

   /* Envoi de l'adresse */
   I2C_RawWrite8 (addr);

   /* Demande d'acquittement */
   result = I2C_GetACK ();

   /* On continue sur une lecture l'acquittement est OK */
   if (!result)
   {
      /* Lecture des 8 bits de poids fort */
      msb = I2C_RawRead8 ();

      /* Envoie un acquittement de lecture */
      I2C_SendACK (1);

      /* Lecture des 8 bits de poids faible */
      lsb = I2C_RawRead8 ();

      /* Envoie un acquittement de lecture */
      I2C_SendACK (0);
   }

   /* Libre le bus I2C */
   I2C_StopCond ();

   /* Valeur de retour */
   return (unsigned short) ((msb << 8) | lsb);
}

/************************************************************************
*                    int I2C_Read16(addr,buffer,size)           PUBLIC  *
*                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                   *
* Objet : Lecture d'un mot de 16 bits  une adresse                     *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  addr : (unsigned char) Adresse de debut de lecture (0-255)           *
*  buffer: (unsigned char *) Pointeur sur la zone mmoire qui recevra   *
*         les octets lus                                                *
*  Size : (int) Nombre d'octets  lire                                  *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (unsigned short) mot lu                                               *
*                                                                       *
************************************************************************/

int
I2C_ReadN (unsigned char addr,
           register unsigned char *buffer, register int size)
{
   /* =======  Variables lcoales */
   int result;                  /* Resultat */

   /* Prise du bus */
   I2C_StartCond ();

   /* Envoi de l'adresse */
   I2C_RawWrite8 (addr);

   /* Demande d'acquittement */
   result = I2C_GetACK ();

   /* On continue sur la lecture si l'acquittement est OK */
   if (!result)
   {
      while (size)
      {
         *buffer = I2C_RawRead8 ();
         buffer++;
         size--;

         /* Envoie un acquittement si lecture non termine */
         I2C_SendACK (size);
      }
   }

   /* Libre le bus I2C */
   I2C_StopCond ();

   /* Valeur de retour */
   return result;
}

/************************************************************************
*                             void I2C_Open()                   PUBLIC  *
*                             ~~~~~~~~~~~~~~~                           *
* Objet : Initialisation du controleur de bus I2C                       *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
I2C_Open (void)
{
   /* Ouverture du port srie */
   rs232_Open ();

   /* Etat initial */
   I2C_CHic = 'R';

   /* Libre le bus I2C */
   I2C_StopCond ();
}

/************************************************************************
*                             void I2C_Close()                  PUBLIC  *
*                             ~~~~~~~~~~~~~~~~                          *
* Objet : Fermeture du controleur de bus I2C                            *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
I2C_Close (void)
{
   /* Libre le bus I2C */
   I2C_StopCond ();

   /* Fermeture du port srie */
   rs232_Close ();
}

/* ######################  FIN DE FICHIER  ########################### */
