/*
##########################################################################
##                                                                      ##
##   RENE GARCIA                          GNU General Public License V2 ##
##                                                                      ##
##########################################################################
##                                                                      ##
##      Projet  : TEMPerHum Linux                                       ##
##      Module  : TEMPer                                                ##
##      Piece   : TEMPer.c                                              ##
##      Langage : C ANSI                                                ##
##      Auteur  : Ren GARCIA                                           ##
##                                                                      ##
##########################################################################
*/

    /*-
     *
     *  $Id: TEMPer.c,v 1.1 2009/10/09 08:03:24 rgarcia Exp $
     *
     *  $Log: TEMPer.c,v $
     *  Revision 1.1  2009/10/09 08:03:24  rgarcia
     *  Version initiale
     *
     *
    -*/

/***************************  Inclusions  *******************************/

#include "I2C.h"
#include "TEMPer.h"
#include "SHT11.h"
#include "DS75.h"
#include "EE2402.h"

/************************  Definitions locales  *************************/

    /*-
     *
     *  Les sondes de la famille TEMPer sont constitues de deux
     *  composants sur un bus I2C : Une EEPROM et un capteur.
     *
     *  L'EEPROM permet d'identifier le type et la version du
     *  priphrique. Sa lecture se fait par des commandes I2C.
     *
     *  Le capteur n'est pas toujours conforme au standard I2C
     *  selon le type de priphrique.
     *
    -*/

#define TYPE_UNKNOWN    0
#define TYPE_TEMPerV1   1       /* Conforme I2C, capteur de temperature */
#define TYPE_TEMPerV2   2       /* Conforme I2C, capteur de temperature plus prcis */
#define TYPE_TEMPerHum  4       /* SHT11, non conforme I2C, capteur de temperature et d'humidit */

/* Adresse des chip DS75 sur le bus I2C (TEMPer V1 et V2) */
#define TEMPER_DS75_ADDR    7

/************************************************************************
*                                                                       *
*        V A R I A B L E S   G L O B A L E S   A U   M O D U L E        *
*                                                                       *
************************************************************************/

/* =======  Type de sonde */
static int TEMPerType = TYPE_UNKNOWN;

/************************************************************************
*                                                                       *
*            P R O T O T Y P E S   D E S   F O N C T I O N S            *
*                                                                       *
************************************************************************/

static void ReadTEMPerType (void);

/************************************************************************
*                       void ReadTEMPerType()                    PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~                           *
* Objet : Identifier le type de priphrique TEMPer ou TEMPerHum        *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

static void
ReadTEMPerType (void)
{
   /* =======  Variables locales */
   unsigned char eeprom[2];

   /* =======  Debut du code */

   /* Lecture de l'eeprom sur 2 octets */
   EE2402_ReadBytes (0x58, eeprom, 2);

   /* Cas 1: TEMPer V1 */
   if ((eeprom[0] == 0x58) & (eeprom[1] == 0x59))
   {
      TEMPerType = TYPE_TEMPerV1;
   }
   /* Cas 2: TEMPer V2 */
   else if ((eeprom[0] == 0x58) & (eeprom[1] != 0x59))
   {
      TEMPerType = TYPE_TEMPerV2;
   }
   /* Cas 3: TEMPerHum */
   else if ((eeprom[0] == 0x59) & (eeprom[1] == 0x5A))
   {
      TEMPerType = TYPE_TEMPerHum;
   }

   /*-
    *
    *  Il existe d'autres type de TEMPer, notamment le TEMperNTC qui
    *  embarque deux capteurs de temprature (intrieur et extrieur)
    *
    *  Il n'est pas support par cet outil faute de documentation
    *  (et faute d'en avoir un pour tester)
    *
   -*/
}

/************************************************************************
*                      void TEMPer_Read(pTemp, pHum)            PUBLIC  *
*                      ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                    *
* Objet : Lecture de la temprature et de l'humidit relative sur un    *
*         TEMPer V1, V2 ou TEMPerHum                                    *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  pTemp : (double) Temperature lue                                     *
*  pHum  : (double) Humidit lue (0.0 sur un TEMPer V1 ou V2)           *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
TEMPer_Read (double *pTemp, double *pHum)
{
   /* valeurs par dfaut */
   *pTemp = -99.0;
   *pHum = -99.0;

   /* Cas d'un TEMPer */
   if ((TEMPerType == TYPE_TEMPerV1) || (TEMPerType == TYPE_TEMPerV2))
   {
      DS75_WriteStatusReg (TEMPER_DS75_ADDR, DS75_RESOLUTION_12BIT);

      DS75_GetTemp (TEMPER_DS75_ADDR, pTemp);
      *pHum = 0;
   }
   /* Cas d'un TEMPerHum */
   else if (TEMPerType == TYPE_TEMPerHum)
   {
      SHT11_GetTempHum (pTemp, pHum);
   }
}

/************************************************************************
*                             void TEMPer_Open()                PUBLIC  *
*                             ~~~~~~~~~~~~~~~~~~                        *
* Objet : Initialisation de la sonde                                    *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
TEMPer_Open (void)
{
   /* Ouverture du bus I2C */
   I2C_Open ();

   /* Determinsation du type de sonde */
   ReadTEMPerType ();
}

/************************************************************************
*                             void TEMPer_Close()               PUBLIC  *
*                             ~~~~~~~~~~~~~~~~~~~                       *
* Objet : Fermeture de la sonde                                         *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

void
TEMPer_Close (void)
{
   /* Fermeture du bus I2C */
   I2C_Close ();
}

/* ######################  FIN DE FICHIER  ########################### */
