/*
##########################################################################
##                                                                      ##
##   RENE GARCIA                          GNU General Public License V2 ##
##                                                                      ##
##########################################################################
##                                                                      ##
##      Projet  : TEMPerHum Linux                                       ##
##      Module  : SHT11                                                 ##
##      Piece   : SHT11.c                                               ##
##      Langage : C ANSI                                                ##
##      Auteur  : Ren GARCIA                                           ##
##                                                                      ##
##########################################################################
*/

    /*-
     *
     *  $Id: SHT11.c,v 1.1 2009/10/09 08:03:24 rgarcia Exp $
     *
     *  $Log: SHT11.c,v $
     *  Revision 1.1  2009/10/09 08:03:24  rgarcia
     *  Version initiale
     *
     *
    -*/

/***************************  Inclusions  *******************************/

#include "rs232.h"
#include "I2C.h"
#include "SHT11.h"
#include "crc.h"

/************************************************************************
*                                                                       *
*               C O N S T A N T E S   D U   M O D U L E                 *
*                                                                       *
************************************************************************/

/* =======  Mots de commande               ADDR  CMD   R/W */
#define SHT11_STATUS_REG_W      0x06    /* 000   0011    0 */
#define SHT11_STATUS_REG_R      0x07    /* 000   0011    1 */
#define SHT11_MEASURE_TEMP      0x03    /* 000   0001    1 */
#define SHT11_MEASURE_HUMI      0x05    /* 000   0010    1 */
#define SHT11_RESET             0x1e    /* 000   1111    0 */

/* =======  Bits du registre d'tat */
#define SHT11_SR_LOWBATT        0x40    /* R/O */
#define SHT11_SR_HEATER         0x04    /* R/W */
#define SHT11_SR_RELOAD         0x02    /* R/W */
#define SHT11_SR_RESOLUTION     0x01    /* R/W */

/************************************************************************
*                                                                       *
*        V A R I A B L E S   G L O B A L E S   A U   M O D U L E        *
*                                                                       *
************************************************************************/

/************************************************************************
*                                                                       *
*            P R O T O T Y P E S   D E S   F O N C T I O N S            *
*                                                                       *
************************************************************************/

static void Delay (register int);
static void SHT11_TransStart (void);
static int SHT11_WriteByte (register unsigned char);
static unsigned char SHT11_ReadByte (char);
static void SHT11_ConnectionReset (void);
static int SHT11_SoftReset (void);
static int SHT11_Mesure (unsigned short *, unsigned char);
static int SHT11_ReadStatusReg (unsigned char *);
static int SHT11_WriteStatusReg (unsigned char);

/************************************************************************
*                            void Delay(msec)                    PRIVE  *
*                            ~~~~~~~~~~~~~~~~                           *
* Objet : Temporisation                                                 *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* msec: (int)  Delai en centimes de milliseconde                       *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/


static void
Delay (register int msec)
{
   usleep (10 * msec);
}


/*#define Delay(x)*/

/************************************************************************
*                       void SHT11_TransStart(void)              PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~                     *
* Objet : Gnrer un signa de debut de transmission pour sensibus       *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

static void
SHT11_TransStart (void)
{
    /*-
     *
     *  Sensibus (non conforme I2C) :
     *  Le signal transmission start :
     *
     *       _____         ________
     * SDA :      |_______|
     *           ___     ___
     * SCL : ___|   |___|   |______
     *
    -*/

   /* Etat Initial */
   I2C_SDout (1);
   I2C_Sclk (0);

   Delay (5);

   I2C_Sclk (1);
   Delay (5);
   I2C_SDout (0);
   Delay (5);
   I2C_Sclk (0);

   Delay (10);

   I2C_Sclk (1);
   Delay (5);
   I2C_SDout (1);
   Delay (5);
   I2C_Sclk (0);
}

/************************************************************************
*                       char SHT11_WriteByte(value)              PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~                     *
* Objet : Envoyer un octet de 8 bits sur le sensibus                    *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  value : (unsigned char) Valeur  envoyer                             *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (char) 0 si tout c'est bien pass, autre valeur sinon                 *
*                                                                       *
************************************************************************/

static int
SHT11_WriteByte (register unsigned char value)
{
   /* Envoi des 8 bits */
   I2C_RawWrite8 (value);

   /* Lecture de l'acquitement */
   return I2C_GetACK ();
}

/************************************************************************
*                       char SHT11_ReadByte(ack)                 PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~                        *
* Objet : Lecture d'un octet de 8 bits sur le sensibus                  *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  ack : (char) 1 s'il faut envoyer un acquittement 0 sinon             *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (unsigned char) valeur lue                                            *
*                                                                       *
************************************************************************/

static unsigned char
SHT11_ReadByte (char ack)
{
   /* =======  Variables locales */
   register unsigned char val = 0;

   /* =======  Debut du code */

   /* Lecture des 8 bits */
   val = I2C_RawRead8 ();

   /* Pose ou pas l'acquittement */
   I2C_SDout (!ack);

   /* 1 cycle d'horloge */
   I2C_HiLowSCLK ();

   /* Liberation de la ligne data */
   I2C_SDout (1);

   return val;
}

/************************************************************************
*                       void SHT11_ConnectionReset(void)         PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                *
* Objet : Reinitialiser les communications du sensibus                  *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (aucune)                                                              *
*                                                                       *
************************************************************************/

static void
SHT11_ConnectionReset (void)
{
   /* =======  Variables locales */
   register unsigned char i;

    /*-
     *
     *  Sensibus (non conforme I2C) :
     *  Le signal reset : 9 cycles horloge avec SDA=1 puis un signal
     *  de debut de transmission
     *
     *       ____________________________________________         ____
     * SDA :                                             |_______|
     *          _   _   _   _   _   _   _   _   _       ___     ___
     * SLC : __| |_| |_| |_| |_| |_| |_| |_| |_| |_____|   |___|   |__
     *
    -*/

   /* =======  Debut du code */

   /* Etat Initial */
   I2C_SDout (1);
   I2C_Sclk (0);

   /* 9 Cycles d'horloge */
   for (i = 0; i < 9; i++)
   {
      I2C_HiLowSCLK ();
   }

   /* Signal de debut de transmission */
   SHT11_TransStart ();
}

/************************************************************************
*                       char SHT11_SoftReset(void)               PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~                      *
* Objet : Reinitialiser la sonde SHT11                                  *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
* (aucun)                                                               *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (char) 0 en cas d'erreur, autre valeur sinon                          *
*                                                                       *
************************************************************************/

static int
SHT11_SoftReset (void)
{
   /* Rinitialisation du bus */
   SHT11_ConnectionReset ();

   /* Envoi de la commande de reset */
   return SHT11_WriteByte (SHT11_RESET);

   /* Temporisation de 11ms */
   usleep (11000);
}

/************************************************************************
*                  char SHT11_ReadStatusReg(pVal,pChk)           PRIVE  *
*                  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                  *
* Objet : Lecture du registre d'tat de la sonde SHT11                  *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  pVal : (unsigned char *) Pointeur sur l'octet qui recevra la valeur  *
*         du registre d'tat                                            *
*  pChk : (unsigned char *) Pointeur sur l'octet qui recevra la valeur  *
*         du checksum                                                   *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout est OK, autre valeur sinon                            *
*                                                                       *
************************************************************************/

static int
SHT11_ReadStatusReg (unsigned char *pValue)
{
   /* =======  Variables locales */
   int status;
   unsigned char checksum;

   /* =======  Debut du code */

   /* Debut de transmission */
   SHT11_TransStart ();
   crc_Init ();

   /* Commande de lecture du registre d'tat */
   status = SHT11_WriteByte (SHT11_STATUS_REG_R);
   crc_Add (SHT11_STATUS_REG_R);

   /* Lecture du registre */
   *pValue = SHT11_ReadByte (1);
   crc_Add (*pValue);

   /* Lecture et test du checksum */
   checksum = SHT11_ReadByte (0);
   status += crc_Check (checksum);

   /* Code retour */
   return status;
}

/************************************************************************
*                  char SHT11_WriteStatusReg(Value)              PRIVE  *
*                  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                     *
* Objet : Ecriture du registre d'tat de la sonde SHT11                 *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  Value : (unsigned char) Valeur  crire dans le registre d'tat      *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout est OK, autre valeur sinon                            *
*                                                                       *
************************************************************************/

static int
SHT11_WriteStatusReg (unsigned char Value)
{
   /* =======  Variables locales */
   int status;

   /* =======  Debut du code */

   /* Debut de transmission */
   SHT11_TransStart ();

   /* Commande d'criture du registre d'tat */
   status += SHT11_WriteByte (SHT11_STATUS_REG_W);

   /* Ecriture du registre */
   status += SHT11_WriteByte (Value);

   /* Code retour */
   return status;
}

/************************************************************************
*                       char SHT11_Mesure(pVal,mode)             PRIVE  *
*                       ~~~~~~~~~~~~~~~~~~~~~~~~~~~~                    *
* Objet : Lecture d'une mesure la sonde SHT11 (valeurs brutes)          *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  pVal : (unsigned short *) Pointeur sur le mot qui recevra la valeur  *
*         de la mesure                                                  *
*  mode : (unsigned char) Valeur a lire 'T'=temperature  'H'=Humidite   *
*         du checksum                                                   *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout est OK, autre valeur sinon                            *
*                                                                       *
************************************************************************/

static int
SHT11_Mesure (unsigned short *pVal, unsigned char mode)
{
   /* =======  Variables locales */

   int status = 0;

   unsigned char checksum,
      msb,
      lsb;

   unsigned int i=0;

   /* =======  Debut du code */

   /* Debut de transmission */
   SHT11_TransStart ();
   crc_Init ();

   /* Commande de lecture du convertisseur choisi */
   switch (mode)
   {
      case 'T':
         status += SHT11_WriteByte (SHT11_MEASURE_TEMP);
         crc_Add (SHT11_MEASURE_TEMP);
         break;

      case 'H':
         status += SHT11_WriteByte (SHT11_MEASURE_HUMI);
         crc_Add (SHT11_MEASURE_HUMI);
         break;

      default:
         status++;
   }

   /* Attente de fin de conversion (5s max) */
   while (I2C_SDin () && (i < 500))
   {
      usleep (10000);
      i++;
   }

   /* Timeout sur conversion */
   if (I2C_SDin ())
   {
      status++;
   }
   else
   {
      /* Lecture de la mesure */
      msb = SHT11_ReadByte (1);
      crc_Add (msb);

      lsb = SHT11_ReadByte (1);
      crc_Add (lsb);

      checksum = SHT11_ReadByte (0);
      status += crc_Check (checksum);

      /* Renvoi de la valeur lue */
      *pVal = (msb << 8) | lsb;
   }

   /* Code de retour */
   return status;
}

/************************************************************************
*                    char SHT11_GetTempHum(pTemp,pHum)          PUBLIC  *
*                    ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                  *
* Objet : Lecture d'une mesure de temperature et d'humidit de la sonde *
*         SHT11 (valeurs corriges)                                     *
*-----------------------------------------------------------------------*
* PARAMETRES                                                            *
*                                                                       *
*  pTemp : (double *) Pointeur qui recevra la mesure de temperature     *
*  pHum: (double *) Pointeur qui recevra la mesure de l'humidite        *
*                                                                       *
*-----------------------------------------------------------------------*
* VALEUR DE RETOUR                                                      *
*                                                                       *
* (int) 0 si tout s'est bien pass, autre valeur sinon                  *
*                                                                       *
************************************************************************/

int
SHT11_GetTempHum (double *pTemp, double *pHum)
{
   /* =======  Variables locales */

   unsigned short humi_val,
      temp_val;

   unsigned int status = 0,
      i;

   /* =======  Debut du code */

   /* Reinitialisation du bus */
   SHT11_ConnectionReset ();

   /* Lecture du capteur d'humidite et de temperature */
   status += SHT11_Mesure (&humi_val, 'H');
   status += SHT11_Mesure (&temp_val, 'T');

   if (status)
   {
      /* Si erreur de lecture, reset du bus */
      SHT11_ConnectionReset ();
   }
   else
   {
      /* Conversion des valeurs des capteurs en valeurs physiques */

      const double C1 = -4.0;           /* pour 12 Bit */
      const double C2 = +0.0405;        /* pour 12 Bit */
      const double C3 = -0.0000028;     /* pour 12 Bit */
      const double T1 = +0.01;          /* pour 14 Bit @ 5V */
      const double T2 = +0.00008;       /* pour 14 Bit @ 5V */

      double rh_lin;
      double rh_true;
      double t_C;

      /* Calcul de la temperature [C] */
      t_C = temp_val * 0.01 - 40.1 - 2E-8 * (temp_val - 7000) * (temp_val - 7000);

      /* Calcul de l'humidit linaire [%RH] */
      rh_lin = C3 * humi_val * humi_val + C2 * humi_val + C1;

      /* Calcul de l'humidit relative [%RH] */
      rh_true = (t_C - 25) * (T1 + T2 * humi_val) + rh_lin;

      /* Borner l'humidite [0.1;100] */
      if (rh_true > 100)
         rh_true = 100;

      if (rh_true < 0.1)
         rh_true = 0.1;

      /* Renvoi des valeurs physiques */
      *pTemp = t_C;
      *pHum = rh_true;
   }

   /* Code de retour */
   return status;
}

/* ######################  FIN DE FICHIER  ########################### */
